/*
* Copyright (c) 1996-1999, 2001-2004 Wolfram Gloger

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that (i) the above copyright notices and this permission
notice appear in all copies of the software and related documentation,
and (ii) the name of Wolfram Gloger may not be used in any advertising
or publicity relating to the software.

THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND,
EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.

IN NO EVENT SHALL WOLFRAM GLOGER BE LIABLE FOR ANY SPECIAL,
INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY
DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
WHETHER OR NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY
OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.
*/

/*
 * $Id: t-test2.c,v 1.3 2004/11/04 15:01:05 wg Exp $
 * by Wolfram Gloger 1996-1999, 2001, 2004
 * A multi-thread test for malloc performance, maintaining a single
 * global pool of allocated bins.
 */

#if (defined __STDC__ && __STDC__) || defined __cplusplus
# include <stdlib.h>
#endif
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <pthread.h>

#if !USE_MALLOC
#include <malloc.h>
#else
#include "malloc.h"
#endif

/* lran2.h
 * by Wolfram Gloger 1996.
 *
 * A small, portable pseudo-random number generator.
 */

#ifndef _LRAN2_H
#define _LRAN2_H

#define LRAN2_MAX 714025l /* constants for portable */
#define IA	  1366l	  /* random number generator */
#define IC	  150889l /* (see e.g. `Numerical Recipes') */

struct lran2_st {
    long x, y, v[97];
};

static void
lran2_init(struct lran2_st* d, long seed)
{
    long x;
    int j;

    x = (IC - seed) % LRAN2_MAX;
    if(x < 0) x = -x;
    for(j=0; j<97; j++) {
	x = (IA*x + IC) % LRAN2_MAX;
	d->v[j] = x;
    }
    d->x = (IA*x + IC) % LRAN2_MAX;
    d->y = d->x;
}

#ifdef __GNUC__
__inline__
#endif
static long
lran2(struct lran2_st* d)
{
    int j = (d->y % 97);

    d->y = d->v[j];
    d->x = (IA*d->x + IC) % LRAN2_MAX;
    d->v[j] = d->x;
    return d->y;
}

#undef IA
#undef IC

#endif

struct user_data {
	int max;
	unsigned long size;
	long seed;
};

#define N_TOTAL		10
#ifndef N_THREADS
#define N_THREADS	2
#endif
#ifndef N_TOTAL_PRINT
#define N_TOTAL_PRINT 50
#endif
#define STACKSIZE	32768
#ifndef MEMORY
#define MEMORY		8000000l
#endif
#define SIZE		10000
#define I_MAX		10000
#define BINS_PER_BLOCK 20

#define RANDOM(d,s)	(lran2(d) % (s))
#define TEST 99

static int verbose = 1;

/* Normal pthread mutex.  */
typedef pthread_mutex_t mutex_t;

#define MUTEX_INITIALIZER          PTHREAD_MUTEX_INITIALIZER
#define mutex_init(m)              pthread_mutex_init(m, NULL)
#define mutex_lock(m)              pthread_mutex_lock(m)
#define mutex_trylock(m)           pthread_mutex_trylock(m)
#define mutex_unlock(m)            pthread_mutex_unlock(m)


/*
 * $Id: thread-st.h$
 * pthread version
 * by Wolfram Gloger 2004
 */

#include <pthread.h>
#include <stdio.h>

pthread_cond_t finish_cond = PTHREAD_COND_INITIALIZER;
pthread_mutex_t finish_mutex = PTHREAD_MUTEX_INITIALIZER;


#ifndef USE_PTHREADS_STACKS
#define USE_PTHREADS_STACKS 0
#endif

#ifndef STACKSIZE
#define STACKSIZE	32768
#endif

struct thread_st {
	char *sp;							/* stack pointer, can be 0 */
	void (*func)(struct thread_st* st);	/* must be set by user */
	pthread_t id;
	int flags;
	struct user_data u;
};

static void
thread_init(void)
{
#if !defined(USE_SYSTEM_MALLOC) && defined(USE_MALLOC_LOCK)
  extern int dnmalloc_pthread_init(void);
  dnmalloc_pthread_init();
#endif

  if (verbose)
	printf("Using posix threads.\n");
  pthread_cond_init(&finish_cond, NULL);
  pthread_mutex_init(&finish_mutex, NULL);
}

static void *
thread_wrapper(void *ptr)
{
	struct thread_st *st = (struct thread_st*)ptr;

	/*printf("begin %p\n", st->sp);*/
	st->func(st);
	pthread_mutex_lock(&finish_mutex);
	st->flags = 1;
	pthread_mutex_unlock(&finish_mutex);
	pthread_cond_signal(&finish_cond);
	/*printf("end %p\n", st->sp);*/
	return NULL;
}

static int
thread_create(struct thread_st *st)
{
	st->flags = 0;
	{
		pthread_attr_t* attr_p = 0;
#if USE_PTHREADS_STACKS
		pthread_attr_t attr;

		pthread_attr_init (&attr);
		if(!st->sp)
			st->sp = malloc(STACKSIZE+16);
		if(!st->sp)
			return -1;
		if(pthread_attr_setstacksize(&attr, STACKSIZE))
			fprintf(stderr, "error setting stacksize");
		else
			pthread_attr_setstackaddr(&attr, st->sp + STACKSIZE);
		/*printf("create %p\n", st->sp);*/
		attr_p = &attr;
#endif
		return pthread_create(&st->id, attr_p, thread_wrapper, st);
	}
	return 0;
}

/* Wait for one of several subthreads to finish. */
static void
wait_for_thread(struct thread_st st[], int n_thr,
				int (*end_thr)(struct thread_st*))
{
	int i;

	pthread_mutex_lock(&finish_mutex);
	for(;;) {
		int term = 0;
		for(i=0; i<n_thr; i++)
			if(st[i].flags) {
				/*printf("joining %p\n", st[i].sp);*/
				if(pthread_join(st[i].id, NULL) == 0) {
					st[i].flags = 0;
					if(end_thr)
						end_thr(&st[i]);
				} else
					fprintf(stderr, "can't join\n");
				++term;
			}
		if(term > 0)
			break;
		pthread_cond_wait(&finish_cond, &finish_mutex);
	}
	pthread_mutex_unlock(&finish_mutex);
}

/* For large allocation sizes, the time required by copying in
   realloc() can dwarf all other execution times.  Avoid this with a
   size threshold. */
#ifndef REALLOC_MAX
#define REALLOC_MAX     2000
#endif

struct bin {
        unsigned char *ptr;
        unsigned long size;
};


struct block {
	struct bin b[BINS_PER_BLOCK];
	mutex_t mutex;
} *blocks;

int n_blocks;
int mem_check(unsigned char *ptr, unsigned long size);
int zero_check(unsigned* ptr, unsigned long size);
void mem_init(unsigned char *ptr, unsigned long size);

static void
bin_alloc(struct bin *m, unsigned long size, int r)
{
#if TEST > 0
        if(mem_check(m->ptr, m->size)) {
                printf("memory corrupt!\n");
                exit(1);
        }
#endif
        r %= 1024;
        /*printf("%d ", r);*/
        if(r < 4) { /* memalign */
                if(m->size > 0) free(m->ptr);
                m->ptr = (unsigned char *)memalign(sizeof(int) << r, size);
        } else if(r < 20) { /* calloc */
                if(m->size > 0) free(m->ptr);
                m->ptr = (unsigned char *)calloc(size, 1);
#if TEST > 0
                if(zero_check((unsigned*)m->ptr, size)) {
                        long i;
                        for(i=0; i<size; i++)
                                if(m->ptr[i] != 0)
                                        break;
                        printf("calloc'ed memory non-zero (ptr=%p, i=%ld)!\n", m->ptr, i);
                        exit(1);
                }
#endif
        } else if(r < 100 && m->size < REALLOC_MAX) { /* realloc */
                if(m->size == 0) m->ptr = NULL;
                m->ptr = realloc(m->ptr, size);
        } else { /* plain malloc */
                if(m->size > 0) free(m->ptr);
                m->ptr = (unsigned char *)malloc(size);
        }
        if(!m->ptr) {
                printf("out of memory (r=%d, size=%ld)!\n", r, (long)size);
                exit(1);
        }
        m->size = size;
#if TEST > 0
		memset(m->ptr, '\0', m->size);
        mem_init(m->ptr, m->size);
#endif
}

/* Free a bin. */

static void
bin_free(struct bin *m)
{
        if(m->size == 0) return;
#if TEST > 0
        if(mem_check(m->ptr, m->size)) {
                printf("memory corrupt!\n");
                exit(1);
        }
#endif
        free(m->ptr);
        m->size = 0;
}


#if TEST > 0

void mem_init(unsigned char *ptr, unsigned long size)
{
        unsigned long i, j;

        if(size == 0) return;
        for(i=0; i<size; i+=2047) {
                j = (unsigned long)ptr ^ i;
                ptr[i] = ((j ^ (j>>8)) & 0xFF);
        }
        j = (unsigned long)ptr ^ (size-1);
        ptr[size-1] = ((j ^ (j>>8)) & 0xFF);
}

int zero_check(unsigned* ptr, unsigned long size)
{
        unsigned char* ptr2;

        while(size >= sizeof(*ptr)) {
                if(*ptr++ != 0)
                        return -1;
                size -= sizeof(*ptr);
        }
        ptr2 = (unsigned char*)ptr;
        while(size > 0) {
                if(*ptr2++ != 0)
                        return -1;
                --size;
        }
        return 0;
}

int
mem_check(unsigned char *ptr, unsigned long size)
{
        unsigned long i, j;

        if(size == 0) return 0;
        for(i=0; i<size; i+=2047) {
                j = (unsigned long)ptr ^ i;
                if(ptr[i] != ((j ^ (j>>8)) & 0xFF)) return 1;
        }
        j = (unsigned long)ptr ^ (size-1);
        if(ptr[size-1] != ((j ^ (j>>8)) & 0xFF)) return 2;
        return 0;
}

void
bin_test(void)
{
	int b, i;

	for(b=0; b<n_blocks; b++) {
		mutex_lock(&blocks[b].mutex);
		for(i=0; i<BINS_PER_BLOCK; i++) {
			if(mem_check(blocks[b].b[i].ptr, blocks[b].b[i].size)) {
				printf("memory corrupt!\n");
				exit(1);
			}
		}
		mutex_unlock(&blocks[b].mutex);
	}
}

#endif

void
malloc_test(struct thread_st *st)
{
	struct block *bl;
	int i, b, r;
	struct lran2_st ld; /* data for random number generator */
	unsigned long rsize[BINS_PER_BLOCK];
	int rnum[BINS_PER_BLOCK];

	lran2_init(&ld, st->u.seed);
	for(i=0; i<=st->u.max;) {
#if TEST > 1
		bin_test();
#endif
		bl = &blocks[RANDOM(&ld, n_blocks)];
		r = RANDOM(&ld, 1024);
		if(r < 200) { /* free only */
			mutex_lock(&bl->mutex);
			for(b=0; b<BINS_PER_BLOCK; b++)
				bin_free(&bl->b[b]);
			mutex_unlock(&bl->mutex);
			i += BINS_PER_BLOCK;
		} else { /* alloc/realloc */
			/* Generate random numbers in advance. */
			for(b=0; b<BINS_PER_BLOCK; b++) {
				rsize[b] = RANDOM(&ld, st->u.size) + 1;
				rnum[b] = lran2(&ld);
			}
			mutex_lock(&bl->mutex);
			for(b=0; b<BINS_PER_BLOCK; b++)
				bin_alloc(&bl->b[b], rsize[b], rnum[b]);
			mutex_unlock(&bl->mutex);
			i += BINS_PER_BLOCK;
		}
#if TEST > 2
		bin_test();
#endif
	}
}

int n_total=0, n_total_max=N_TOTAL, n_running;

int
my_end_thread(struct thread_st *st)
{
	/* Thread st has finished.  Start a new one. */
#if 0
	printf("Thread %lx terminated.\n", (long)st->id);
#endif
	if(n_total >= n_total_max) {
		n_running--;
	} else if(st->u.seed++, thread_create(st)) {
		printf("Creating thread #%d failed.\n", n_total);
	} else {
		n_total++;
		if(n_total%N_TOTAL_PRINT == 0)
			printf("n_total = %d\n", n_total);
	}
	return 0;
}

int
main(int argc, char *argv[])
{
	int i, j, bins;
	int n_thr=N_THREADS;
	int i_max=I_MAX;
	unsigned long size=SIZE;
	struct thread_st *st;

#if USE_MALLOC && USE_STARTER==2
	ptmalloc_init();
	printf("ptmalloc_init\n");
#endif

	if(argc > 1) n_total_max = atoi(argv[1]);
	if(n_total_max < 1) n_thr = 1;
	if(argc > 2) n_thr = atoi(argv[2]);
	if(n_thr < 1) n_thr = 1;
	if(n_thr > 100) n_thr = 100;
	if(argc > 3) i_max = atoi(argv[3]);

	if(argc > 4) size = atol(argv[4]);
	if(size < 2) size = 2;

	bins = MEMORY/size;
	if(argc > 5) bins = atoi(argv[5]);
	if(bins < BINS_PER_BLOCK) bins = BINS_PER_BLOCK;

	n_blocks = bins/BINS_PER_BLOCK;
	blocks = (struct block *)malloc(n_blocks*sizeof(*blocks));
	if(!blocks)
		exit(1);

	thread_init();
	printf("total=%d threads=%d i_max=%d size=%ld bins=%d\n",
		   n_total_max, n_thr, i_max, size, n_blocks*BINS_PER_BLOCK);

	for(i=0; i<n_blocks; i++) {
		mutex_init(&blocks[i].mutex);
		for(j=0; j<BINS_PER_BLOCK; j++) blocks[i].b[j].size = 0;
	}

	st = (struct thread_st *)malloc(n_thr*sizeof(*st));
	if(!st) exit(-1);

#if !defined NO_THREADS && (defined __sun__ || defined sun)
	/* I know of no other way to achieve proper concurrency with Solaris. */
	thr_setconcurrency(n_thr);
#endif

	/* Start all n_thr threads. */
	for(i=0; i<n_thr; i++) {
		st[i].u.max = i_max;
		st[i].u.size = size;
		st[i].u.seed = ((long)i_max*size + i) ^ n_blocks;
		st[i].sp = 0;
		st[i].func = malloc_test;
		if(thread_create(&st[i])) {
			printf("Creating thread #%d failed.\n", i);
			n_thr = i;
			break;
		}
		printf("Created thread %lx.\n", (long)st[i].id);
	}

	for(n_running=n_total=n_thr; n_running>0;) {
		wait_for_thread(st, n_thr, my_end_thread);
	}

	for(i=0; i<n_blocks; i++) {
		for(j=0; j<BINS_PER_BLOCK; j++)
			bin_free(&blocks[i].b[j]);
	}

	for(i=0; i<n_thr; i++) {
		free(st[i].sp);
	}
	free(st);
	free(blocks);
#if USE_MALLOC
	malloc_stats();
#endif
	printf("Done.\n");
	return 0;
}

/*
 * Local variables:
 * tab-width: 4
 * End:
 */
